/* vm.c
 *
 * Copyright (c) 1998 by Fredrik Noring.
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <arpa/inet.h>

/* Virtual machine operators. */
#define OP_STOP     0
#define OP_LDI      1
#define OP_LDL      2
#define OP_LDG      3
#define OP_POP      4
#define OP_POPNPUSH 5
#define OP_APPLY    6

/* Definitions. */
#define ONE_K     1024
#define ONE_MEG   (ONE_K*ONE_K)

#define HEAP_SIZE ONE_MEG

int hp, sp, pc, heap[HEAP_SIZE];

/* Run-time system messages. */
char *say(char *msg, ...)
{
  va_list ap;
  
  va_start(ap, msg);
  vfprintf(stdout, msg, ap);
  va_end(ap);
  return msg;
}

char *err(char *msg, ...)
{
  va_list ap;
  
  va_start(ap, msg);
  fprintf(stderr, "#Error: ");
  vfprintf(stderr, msg, ap);
  va_end(ap);
  return msg;
}

/* Program operators. */
int pop_op()
{
  return heap[pc++];
}

/* Stack operators. */
void push_stack(int svalue)
{
  heap[--sp] = svalue;
}

int pop_stack()
{
  return heap[sp++];
}

/* Heap operators. */
void push_heap(int hvalue)
{
  heap[hp] = hvalue;
}

/* Built in functions. */

void bif_plus(int argc)
{
  int a = 0;
  while(argc--)
    a += pop_stack();
  push_stack(a);
}

void bif_display(int argc)
{
  while(argc--)
    say("%d ", pop_stack());
  say("\n");
}

int global_counter = 2;
int global_symbols[ONE_K] = { (int) bif_plus, (int) bif_display };

int fetch_global(int n)
{
  return global_symbols[n];
}

int fetch_local(int frame, int n)
{
  return heap[sp+n];
}

/* Virtual machine. */
void vm()
{
  int op, i, x;
  
  while(1) {
    op = pop_op();
    switch(op) {
    case OP_STOP:
      return;
    case OP_LDI:
      push_stack(pop_op());
      break;
    case OP_LDL:
      i = pop_op();
      x = pop_op();
      push_stack(fetch_local(i, x));
      break;
    case OP_LDG:
      push_stack(fetch_global(pop_op()));
      break;
    case OP_POP:
      pop_stack();
      break;
    case OP_POPNPUSH:
      x = pop_stack();
      for(i = pop_op(); i--; )
	pop_stack();
      push_stack(x);
      break;
    case OP_APPLY:
      i = pop_op();
      if(i)
	(*((void(*)()) pop_stack()))(i);
      break;
    default:
      err("Unknown operation. [op: %d. pc: %d]", op, pc);
      exit(1);
    }
  }
}

/* Initialize virtual machine. */
void bootstrap()
{
  hp = 0;
  sp = HEAP_SIZE;
}

/* Load program code. */
void load_program(char *filename)
{
  FILE *fp;
  int n[4];

  pc = hp;
  fp = fopen(filename, "r");
  if(!fp)
    return;
  while(((n[0] = getc(fp)) != EOF) &&
	((n[1] = getc(fp)) != EOF) &&
	((n[2] = getc(fp)) != EOF) &&
	((n[3] = getc(fp)) != EOF)) {
    heap[hp] = (int) ntohl((n[0] << 24) | (n[1] << 16) | (n[2] << 8) | n[3]);
    hp++;
  }
  fclose(fp);
}

int main(int argc, char **argv)
{
  bootstrap();
  load_program("a.bc");
  vm();
  
  return 0;
}
