/* libshoe.c
 *
 * Copyright (c) 1998 by Fredrik Noring.
 *
 * This is the entire Shoe interpreter and runtime system.
 */

#include <stdio.h>
#include <string.h>
#include <arpa/inet.h>

#include "libshoe.h"

#define T                memdup("#t")
#define F                memdup("#f")
#define ERR              (trace?panic("#ERR."):memdup("#ERR."))
#define SYMBOL_SPACE()   { if(symbol_counter >= MAX_SYMBOLS) panic("Symbol table full."); }
#define FP(x)            MATCH((x), F)
#define TP(x)            (!FP(x))
#define NILP(x)          MATCH((x), "()")
#define LISTP(x)         ((x)[0] == '(')
#define MATCH(a, b)      (strcmp((a), (b)) == 0)
#define DIGITP(c)        ((c) >= '0' && (c) <= '9')
#define BALANCE(c)       ((((c) == '(')?1:0)-(((c) == ')')?1:0))
#define EVAL(x)          eval(car(x))
#define EVALARG(x)       EVAL(cdr(x))
#define ERRP(x)          ((x)[0] == '#' && (x)[1] == 'E')
#define DEVAL(s, op) \
  char *a, *b; \
  s = push_stack(s); a = push_stack(EVAL(s)); \
  b = push_stack(EVALARG(s)); op; pop_n_elems(3); return s;
#define NUM_OP(op, name) char* name(char* s) \
  { DEVAL(s, sprintf(s = mem(16), "%ld", atol(a) op atol(b))); }
#define TIME_TO_GC       (stack_pointer-heap_pointer < GC_MINIMUM)

char *symbols[MAX_SYMBOLS][2];
char *heap_pointer, *stack_pointer, heap[HEAP_SIZE];
int online = 0, trace = 0, symbol_counter, bounded_counter;

long atol(char *);

int spacep(char c)
{
  return c == ' '|| c == '\t'|| c == '\n' || c == '\r';
}

char *panic(char *msg)
{
  fprintf(stderr, "\n#Panic. %s\n", msg);
  exit(1);
  return msg;
}

char *mem(int amount)
{
  if(heap_pointer+amount > stack_pointer)
    panic("Out of memory.");
  return (heap_pointer += amount) - amount;
}

char *memdup(char *s)
{
  return strcpy(mem(strlen(s)+1), s);
}

char *exterr(char *msg, char *context)
{
  fprintf(stderr, "#Exception. %s%s\n", msg, context?context:"#no-context?");
  return ERR;
}

char *push_stack(char *s)
{
  return strcpy(stack_pointer -= strlen(s)+1, s);
}

void pop_n_elems(int n)
{
  while(n--)
    stack_pointer += strlen(stack_pointer)+1;
}

char *gc()
{
  char *chunk, *minimum;
  int s_j = 0, s_k = 0, i, j, k;

  minimum = heap;
  for(i = 0; i < 2*symbol_counter; i++) {
    chunk = heap+HEAP_SIZE;
    for(j = 0; j < symbol_counter; j++)
      for(k = 0; k < 2; k++)
	if(symbols[j][k] <= chunk && minimum <= symbols[j][k])
	  chunk = symbols[s_j=j][s_k=k];
    symbols[s_j][s_k] = minimum;
    while(*chunk)
      *minimum++ = *chunk++;
    *minimum++ = '\0';
  }
  heap_pointer = minimum;
  return T;
}

void check_gc()
{
  if(TIME_TO_GC)
    gc();
}

char *trim(char *s)
{
  while(spacep(*s))
    s++;
  return s;
}

char *suf(char *a, char *b)
{
  char *s;
  sprintf(s = mem(strlen(a)+strlen(b)+1), "%s%s", a, b);
  return s;
}

int statement_size(char* s)
{
  char *source;
  int nbalance = 0;
  
  source = s = trim(s);
  while(nbalance | !((spacep(*s) | (*s == ')')) || (*s == '(' && s-source))) {
    nbalance += BALANCE(*s);
    s++;
  }
  return s-source;
}

char *car(char* s)
{
  int size;
  
  if(!LISTP(s)) return exterr("Cannot car: ", s);
  if(NILP(s)) return s;
  size = statement_size(++s);
  s = strncpy(mem(size+1), s, size);
  s[size] = '\0';
  return s;
}

char *cdr(char *s)
{
  if(!LISTP(s)) return exterr("Cannot cdr: ", s);
  s = trim(s+statement_size(++s));
  s = strcpy(mem(strlen(s)+2)+1, s)-1;
  s[0] = '(';
  return s;
}

char *bind(char *symbol, char *value)
{
  SYMBOL_SPACE();
  bounded_counter++;
  symbols[symbol_counter][1] = value;
  return symbols[symbol_counter++][0] = symbol;
}

char *bif_cons(char *s)
{
  DEVAL(s, (NILP(b)?sprintf(s = mem(strlen(a)+3), "(%s)", a):
	    LISTP(b)?sprintf(s = mem(strlen(a)+strlen(b)+3), "(%s %s", a, b+1):
	    sprintf(s = mem(strlen(a)+strlen(b)+4), "(%s %s)", a, b)));
}

char *bif_lambda(char *s)
{
  return suf("#lambda ", s);
}

char *bif_macro(char *s)
{
  return suf("#macro ", s);
}

char *bif_car(char *s)
{
  return car(EVAL(s));
}

char *bif_cdr(char *s)
{
  return cdr(EVAL(s));
}

char *bif_if(char *s)
{
  s = EVALARG(FP(EVAL(s = push_stack(s)))?cdr(s):s);
  pop_n_elems(1);
  return s;
}

char *bif_equal(char *s)
{
  DEVAL(s, s = MATCH(a, b)?T:F);
}

char *bif_function(char* s)
{
  return EVAL(s);
}

char *bif_print(char* s)
{
  printf("%s", s = EVAL(s));
  return s;
}

char *bif_eval(char* s)
{
  return eval(EVAL(s));
}

char *bif_listp(char* s)
{
  return LISTP(EVAL(s))?T:F;
}

char *bif_numberp(char* s)
{
  return DIGITP(EVAL(s)[0])?T:F;
}

char *bif_less_than(char *s)
{
  DEVAL(s, s = atol(a)<atol(b)?T:F);
}

char *bif_trace(char *s)
{
  s = car(s);
  return (trace=TP(s))?T:F;
}

char *bif_define(char *s)
{
  return bind(car(s), NILP(cdr(cdr(s)))?eval(car(cdr(s))):suf("#lambda ", cdr(s)));
}

char *bif_memory(char *s)
{
  sprintf(s = mem(64), "((heap %lu) (stack %lu) (available %lu) (total %lu))",
	  (unsigned long) (heap_pointer-heap),
	  (unsigned long) (heap+HEAP_SIZE-stack_pointer),
	  (unsigned long) (stack_pointer-heap_pointer),
	  (unsigned long) (HEAP_SIZE));
  return s;
}

NUM_OP(+, bif_plus)
NUM_OP(*, bif_multiply)

char *bif_minus(char *s)
{
  DEVAL(s, sprintf(s = mem(16), "%ld", NILP(b)?-atol(a):atol(a)-atol(b)));
}

char *bif_divide(char *s)
{
  DEVAL(s, sprintf(s = mem(16), "%ld", atol(b)==0?0:atol(a)/atol(b)));
}

FILE *fp;
int loader()
{
  return getc(fp);
}

char *bif_load(char *s)
{
  fp = fopen(car(s), "r");
  if(!fp) return F;
  do s = reader(loader); while(s && eval(s));
  fclose(fp);
  return T;
}

char *bif_read(char *s)
{
  fp = fopen(car(s), "r");
  if(!fp) return F;
  *(s = mem(1)) = '(';
  while(reader(loader));
  *mem(1) = ')';
  fclose(fp);
  return s;
}

char *bif_write(char *s)
{
  fp = fopen(car(s), "wt");
  if(!fp) return F;
  s = EVALARG(s)+1;
  while(*s && *s != ')') {
    char *b;
    long n, i;
    
    while(spacep(*s)) s++;
    b = s;
    if(DIGITP(*s)) {
      while(DIGITP(*s)) s++;
      *s++ = '\0';
      n = htonl(atol(b));
      for(i = 4; i--; )
	fputc((n >> (8*i))&0xff, fp);
    } else if(*s) s++;
  }
  fclose(fp);
  return T;
}

char *eval(char *s)
{
  char macro, *args, *vars, *body;
  int rest = 0, i, old_symbol_counter, old_bounded_counter;

  if(!s) exit(0);
  if(!online) bootstrap();
  if(trace) printf("#eval: [%s]\n", s);
  s = trim(s);
  if(strlen(s) == 0 || s[0] == '#' || DIGITP(*s) || NILP(s))
    return s;
  for(i = bounded_counter; i--; )
    if(MATCH(s, symbols[i][0])) {
      if(trace) printf("#-> value: [%s = %s]\n", s, symbols[i][1]);
      return symbols[i][1];
    }

  s = push_stack(s);
  check_gc();
  body = push_stack(EVAL(s));
  args = push_stack(cdr(s));
  if(body[0] == '#' && (body[1] == 'l' || body[1] == 'm')) {
    macro = body[1]=='m';
    body += (macro?7:8);
    old_symbol_counter = symbol_counter;
    old_bounded_counter = bounded_counter;
    vars = push_stack(car(body));
    while(!ERRP(vars) && !ERRP(args) && (!NILP(args) || !NILP(vars))) {
      s = memdup(macro?car(args):EVAL(args));
      if(rest) {
	char *t;
	t = symbols[rest][1];
	t[strlen(t)-1] = '\0';
	symbols[rest][1] = suf(suf(suf(t, " "), s), ")");
      } else {
	if(MATCH(car(vars), "#rest")) {
	  s = NILP(args)?memdup("()"):suf(suf("(", s), ")");
	  vars = cdr(vars);
	  rest = symbol_counter;
	}
	SYMBOL_SPACE();
	symbols[symbol_counter][1] = s;
	symbols[symbol_counter++][0] = memdup(car(vars));
      }
      vars = cdr(vars);
      args = cdr(args);
      pop_n_elems(2);
      vars = push_stack(vars);
      args = push_stack(args);
    }
    bounded_counter = symbol_counter;
    s = EVALARG(body);
    pop_n_elems(4);
    symbol_counter = old_symbol_counter;
    bounded_counter = old_bounded_counter;
    return macro?eval(s):s;
  }
  if(body[0] != '#' || !DIGITP(body[1])) return exterr("Cannot call: ", body);
  s = ERRP(body)?memdup(body):(*((char*(*)())atol(body+1)))(args);
  pop_n_elems(3);
  return s;
}

void bif(char *symbol, void *f)
{
  bounded_counter++;
  SYMBOL_SPACE();
  sprintf(symbols[symbol_counter][0] = mem(strlen(symbol)+1), "%s", symbol);
  sprintf(symbols[symbol_counter++][1] = mem(17), "#%ld", (unsigned long) f);
}

char *reader(int (*r)(void))
{
  char *s, *source;
  int c, nbalance = 0;
  
  if(!online) bootstrap();
  s = source = mem(1);
  while((nbalance > 0) | !spacep(c = r())) {
    if(c == ';') while((c = r()) != '\n' && c != EOF);
    else if(c == '{') while((c = r()) != '}' && c != EOF);
    else {
      nbalance += BALANCE(c);
      *s++ = c;
      mem(1);
    }
    if(c == EOF) return 0;
  }
  *s = '\0';
  return trim(source);
}

void bootstrap()
{
  heap_pointer = heap;
  stack_pointer = heap+HEAP_SIZE;
  symbol_counter = bounded_counter = 0;
  bif("eval",     bif_eval);
  bif("load",     bif_load);
  bif("function", bif_function);
  bif("quote",    car);
  bif("lambda",   bif_lambda);
  bif("macro",    bif_macro);
  bif("define",   bif_define);
  bif("if",       bif_if);
  bif("equal",    bif_equal);
  bif("<",        bif_less_than);
  bif("+",        bif_plus);
  bif("-",        bif_minus);
  bif("*",        bif_multiply);
  bif("/",        bif_divide);
  bif("car",      bif_car);
  bif("cdr",      bif_cdr);
  bif("cons",     bif_cons);
  bif("memory",   bif_memory);
  bif("trace",    bif_trace);
  bif("list?",    bif_listp);
  bif("number?",  bif_numberp);
  bif("print",    bif_print);
  bif("read",     bif_read);
  bif("write",    bif_write);
  bif("gc",       gc);
  online = 1;
}
